/*!Copyright(c) 2010-2014 Shenzhen TP-LINK Technologies Co.Ltd.
 *
 *\file     mc_list.h
 *\brief    
 *
 *\author   liang liang
 *\version  1.0.0
 *\date     2014-07-05
 *
 *\history 
 *                  
 *          \arg 1.0.0, 2014-07-05, liang liang, Create the file.
 */

#ifndef __MC_LIST_H_
#define __MC_LIST_H_

/*
 ****************************************************************************
 * 1                    DESCRIPTION
 ****************************************************************************
 * Interpeak implementation of a doubly linked list.
 */

/*
 ****************************************************************************
 * 2                    CONFIGURATION
 ****************************************************************************
 */


/*
 ****************************************************************************
 * 3                    INCLUDE FILES
 ****************************************************************************
 */

#include "os_type.h"

#ifdef __cplusplus
extern "C" {
#endif

/*
 ****************************************************************************
 * 4                    DEFINES
 ****************************************************************************
 */

/*
 ****************************************************************************
 * 5                    TYPES
 ****************************************************************************
 */

/*
 *===========================================================================
 *                         mc_list
 *===========================================================================
 * List head and entry structure.
 */
typedef struct _mc_list
{
  struct _mc_list   *next;
  struct _mc_list   *prev;
  struct _mc_list   *head;
  os_u32 size;
}mc_list;


/*
 *===========================================================================
 *                      mc_list_entry
 *===========================================================================
 * Description: Returns the element that contains the mc_list
 *                      member. NOTE: it is not strictly necessary to
 *                      use this macro if the mc_list entry is the
 *                      first member of the containing type, but it is
 *                      good practice to do so even in that case.
 * Parameters:  node - pointer returned from mc_list_first(),
 *                      mc_list_last() or mc_list_next()
 *              type - type of the structure that contains the
 *                      mc_list node.
 *              list_member - name of the member in the 'type'
 *                      structure that is mc_list.
 * Returns:     The element that contains the list member.
 *
 */
#define mc_list_entry(node, type, list_member)   \
    ((type *)((char*)(node) - ip_offsetof(type, list_member)))


/*
 ****************************************************************************
 * 6                    FUNCTIONS
 ****************************************************************************
 */

/*
 *===========================================================================
 *                    mc_list_init
 *===========================================================================
 * Description:   Initialize a list header.
 * Parameters:    head - The uninitialized list head pointer.
 * Returns:       .
 *
 */
void mc_list_init(mc_list *head);


/*
 *===========================================================================
 *                         mc_list_first
 *===========================================================================
 * Description:   Get the first list entry in list 'head'.
 * Parameters:    head - Pointer to the list header.
 * Returns:       Pointer to the first entry in the list or IP_NULL if list
 *                is empty.
 *
 */
mc_list *mc_list_first(mc_list *head);

#define MC_LIST_FIRST(xxhead)   (void *)mc_list_first(xxhead)


/*
 *===========================================================================
 *                         mc_list_last
 *===========================================================================
 * Description:   Get the last list entry in list 'head'.
 * Parameters:    head - Pointer to the list header.
 * Returns:       Pointer to the last entry in the list or IP_NULL if list
 *                is empty.
 *
 */
mc_list *mc_list_last(mc_list *head);

#define MC_LIST_LAST(xxhead)   (void *)mc_list_last(xxhead)


/*
 *===========================================================================
 *                         mc_list_next
 *===========================================================================
 * Description:   Get the next list entry following 'entry'.
 * Parameters:    head - Pointer to the list entry.
 * Returns:       Pointer to the next entry in the list or IP_NULL if none.
 *
 */
mc_list *mc_list_next(mc_list *entry);

#define MC_LIST_NEXT(xxentry)   (void *)mc_list_next(xxentry)


/*
 *===========================================================================
 *                         mc_list_prev
 *===========================================================================
 * Description:   Get the previous list entry to 'entry'.
 * Parameters:    head - Pointer to the list entry.
 * Returns:       Pointer to the previous entry or IP_NULL if none.
 *
 */
mc_list *mc_list_prev(mc_list *entry);

#define MC_LIST_PREV(xxentry)   (void *)mc_list_prev(xxentry)


/*
 *===========================================================================
 *                    mc_list_insert_first
 *===========================================================================
 * Description:   Insert list 'entry' first in the list 'head'.
 * Parameters:    head  - Pointer to the list header to insert 'entry' in.
 *                entry - Pointer to the list entry to insert first.
 * Returns:       .
 *
 */
void mc_list_insert_first(mc_list *head, mc_list *entry);


/*
 *===========================================================================
 *                    mc_list_insert_last
 *===========================================================================
 * Description:   Insert list 'entry' last in the list 'head'.
 * Parameters:    head  - Pointer to the list header to insert 'entry' in.
 *                entry - Pointer to the list entry to insert last.
 * Returns:       .
 *
 */
void mc_list_insert_last(mc_list *head, mc_list *entry);


/*
 *===========================================================================
 *                    mc_list_insert_before
 *===========================================================================
 * Description:   Insert the new list 'entry' before the entry 'inlist' which
 *                is already in a list.
 * Parameters:    inlist - Pointer to the list entry already in a list.
 *                entry  - Pointer to the new list entry to insert before
 *                         'inlist'.
 * Returns:       .
 *
 */
void mc_list_insert_before(mc_list *inlist, mc_list *entry);


/*
 *===========================================================================
 *                    mc_list_insert_after
 *===========================================================================
 * Description:   Insert the new list 'entry' after the entry 'inlist' which
 *                is already in a list.
 * Parameters:    inlist - Pointer to the list entry already in a list.
 *                entry  - Pointer to the new list entry to insert after
 *                         'inlist'.
 * Returns:       .
 *
 */
void mc_list_insert_after(mc_list *inlist, mc_list *entry);


/*
 *===========================================================================
 *                    mc_list_remove
 *===========================================================================
 * Description:   Remove list entry pointer to by 'entry' from its list.
 * Parameters:    entry - Pointer to the list entry to remove.
 * Returns:       .
 *
 */
void mc_list_remove(mc_list *entry);


/*
 *===========================================================================
 *                    mc_list_unlink_head
 *===========================================================================
 * Description:   Unlink 'head' and make the first entry the new 'head'.
 * Parameters:    head  - Pointer to the list header to unlink.
 * Returns:       Pointer to the new head or IP_NULL if list was empty.
 *
 */
mc_list *mc_list_unlink_head(mc_list *head);


/*
 *===========================================================================
 *                    mc_list_size
 *===========================================================================
 * Description:   Get the entry number in list 'head'.
 * Parameters:    head  - Pointer to the list header.
 * Returns:       entry num.
 *
 */
os_u32
mc_list_size(mc_list *head);

#ifdef __cplusplus
}
#endif

#endif

/*
 ****************************************************************************
 *                      END OF FILE
 ****************************************************************************
 */

